/*
 * Routines for dealing with topo maps
 */
#include <sys/types.h>
#ifndef _WIN32
#include <netinet/in.h>
#endif
#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_topo_map.h"
#include "lf_lag.h"

/*
 * Create the topological representation of the fabric in a form suitable
 * for sending
 */
struct lf_topo_map *
lf_create_topo_map(
  struct lf_fabric *fp,
  int *ret_map_size)
{
  int num_nics;
  int num_links;
  int num_links_down;
  int topo_map_size;
  void *tm_buf;
  struct lf_topo_map *topo_map;
  struct lf_topo_nic *topo_nic;
  struct lf_topo_xbar *topo_xbar;
  struct lf_topo_link *topo_link;
  lf_topo_link_state_t *topo_link_state;
  struct lf_lag_id_def *lidp;
  unsigned char *lap;
  struct lf_xbar *xp;
  struct lf_host *hp;
  struct lf_nic *nicp;
  union lf_node *onp;
  uint32_t node_id;
  int node_index;
  int hostname_array_size;
  char *hnp;
  int tn;
  int tx;
  int tl;
  int l;
  int p;
  int n;
  int x;
  int h;

  /* If no hosts or xbars, then no need to proceed with a map */
  if (fp->num_hosts == 0) {
    return NULL;
  }

  /*
   * we need to make 2 passes through the fabric - once to count everything
   * so we can allocate space, then once more to copy things into the
   * topo_map struct
   */

  /* count all links, counting each only once.  this is a little tricky in
   * the case of an xbarport connected back to itself, make sure not to count
   * that as only half a link.
   * The rule is this: visit all xbar ports.  If connected, count the link iff:
   * 	the other node is not an xbar  OR
   * 	the other node is an xbar with an address > this xbar  OR
   * 	the other node is this same xbar, and the port # is >= current port #
   */
  num_links = 0;
  num_links_down = 0;
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    int p;

    xp = fp->xbars[x];

    for (p=0; p<xp->num_ports; ++p) {
      struct lf_xbar *oxp;

      oxp = LF_XBAR(xp->topo_ports[p]);
      if (oxp == NULL
	  || (oxp->ln_type == LF_NODE_XBAR && oxp < xp)
	  || (oxp == xp && xp->topo_rports[p] < p)) {
	continue;
      }

      ++num_links;
    }
  }

  /* repeat basically same rule for counting NIC-NIC links */
  num_nics = 0;
  hostname_array_size = 0;
  for (h=0; h<fp->num_hosts; ++h) {

    hp = fp->hosts[h];
    num_nics += hp->num_nics;	/* count the NICs while we're here */
    hostname_array_size += strlen(hp->hostname)+1;

    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      for (p=0; p<nicp->num_ports; ++p) {
	struct lf_nic *onicp;

	onicp = LF_NIC(nicp->topo_ports[p]);
	if (onicp == NULL
	    || onicp->ln_type != LF_NODE_NIC
	    || onicp < nicp
	    || (onicp == nicp && nicp->topo_rports[p] < p)) {
	  continue;
	}

	++num_links;
      }
    }
  }

  /*
   * Everything is counted, allocate the space for the topo map
   */
  topo_map_size = LF_TOPO_MAP_SIZE(fp->num_xbars, num_nics, num_links,
      fp->num_lag_ids, hostname_array_size);
  LF_CALLOC(tm_buf, char, topo_map_size);

  /* fill in map header */
  topo_map = tm_buf;
  topo_map->num_hosts_32 = htonl(fp->num_hosts);
  topo_map->num_xbars_32 = htonl(fp->num_xbars);
  topo_map->num_nics_32 = htonl(num_nics);
  topo_map->num_links_32 = htonl(num_links);
  topo_map->num_lag_ids_32 = htonl(fp->num_lag_ids);
  topo_map->hostname_array_size_32 = htonl(hostname_array_size);

  /* fill in all the xbars */

  topo_xbar = LF_TOPO_XBAR_ARRAY(tm_buf);
  tx = 0;

  for (x=0; x<fp->num_xbars; ++x) {

    xp = fp->xbars[x];

    node_id = LF_TOPO_XBAR_NODE_ID(x);
    xp->x_topo_index = tx;
    topo_xbar[tx].node_id_32 = htonl(node_id);
    topo_xbar[tx].xbar_id_32 = htonl(xp->xbar_id);
    topo_xbar[tx].num_ports_16 = htons(xp->num_ports);
    topo_xbar[tx].clos_level_8 = xp->clos_level;
    topo_xbar[tx].quadrant_disable_8 = xp->quadrant_disable;

    ++tx;
  }
  if (tx != fp->num_xbars) LF_ERROR(("xbar count mismatch!"));

  /*
   * fill in all the NICs 
   */
  topo_nic = LF_TOPO_NIC_ARRAY(tm_buf);
  hnp = LF_TOPO_HOSTNAME_ARRAY(tm_buf);
  tn = 0;
  for (h=0; h<fp->num_hosts; ++h) {

    hp = fp->hosts[h];

    /* copy hostname into packed array */
    strcpy(hnp, hp->hostname);
    hnp += strlen(hnp)+1;

    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      node_id = LF_TOPO_NIC_NODE_ID(h, nicp->host_nic_id);
      nicp->n_topo_index = tn;
      topo_nic->node_id_32 = htonl(node_id);
      memcpy(topo_nic->mac_addr, nicp->mac_addr, sizeof(lf_mac_addr_t));
      topo_nic->num_ports_8 = nicp->num_ports;
      topo_nic->firmware_type_8 = nicp->host->fw_type;
      topo_nic->flags_16 = htons(nicp->host->fma_flags);
      topo_nic->map_distributor_16 = htons(nicp->host->distributor);
      topo_nic->subfab_mask_32 = htonl(nicp->host->subfabric_mask);
      topo_nic->partition_16 = htons(nicp->partition);
      topo_nic->mag_id_16 = htons(nicp->mag_id);
      topo_nic->lag_id_index_16 = htons(nicp->lag_id_index);

      ++topo_nic;
      ++tn;
    }
  }
  if (tn != num_nics) LF_ERROR(("NIC count mismatch!"));

  /*
   * Fill in links.  First the xbar-based links, then NIC-NIC links
   */
  topo_link = LF_TOPO_LINK_ARRAY(tm_buf);
  topo_link_state = LF_TOPO_LINK_STATE_ARRAY(tm_buf);
  tl = 0;

  for (x=0; x<fp->num_xbars; ++x) {
    xp = fp->xbars[x];

    for (p=0; p<xp->num_ports; ++p) {

      onp = xp->topo_ports[p];
      if (onp == NULL
	  || (onp->ln_type == LF_NODE_XBAR && onp < LF_NODE(xp))
	  || (onp == LF_NODE(xp) && xp->topo_rports[p] < p)) {
	continue;
      }

      /* save this xbar as node_index[0] */
      node_index = LF_TOPO_XBAR_INDEX(xp->x_topo_index, p);
      topo_link->node_index_32[0] = htonl(node_index);

      /* fill in node_index[1] with link partner */
      if (onp->ln_type == LF_NODE_XBAR) {
	struct lf_xbar *oxp;

	oxp = LF_XBAR(onp);
	node_index = LF_TOPO_XBAR_INDEX(oxp->x_topo_index,
				       xp->topo_rports[p]);
	topo_link->node_index_32[1] = htonl(node_index);

      } else if (onp->ln_type == LF_NODE_NIC) {
	struct lf_nic *onicp;

	onicp = LF_NIC(onp);
	node_index = LF_TOPO_NIC_INDEX(onicp->n_topo_index,
				      xp->topo_rports[p]);
	topo_link->node_index_32[1] = htonl(node_index);

      } else {
	LF_ERROR(("Bad type in topo_ports!"));
      }

      /* Save link state and topo index */
      lf_set_xbar_link_topo_index(xp, p, tl);
      topo_link_state[tl] = xp->link_state[p];

      ++topo_link;
      ++tl;
    }
  }

  /* loop through all NICs for NIC-NIC links */
  for (h=0; h<fp->num_hosts; ++h) {

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      for (p=0; p<nicp->num_ports; ++p) {
	struct lf_nic *onicp;

	onp = nicp->topo_ports[p];
	if (onp == NULL
	    || onp->ln_type != LF_NODE_NIC
	    || onp < LF_NODE(nicp)
	    || (onp == LF_NODE(nicp) && nicp->topo_rports[p] < p)) {
	  continue;
	}

	/* save this NIC as node_index[0] */
	node_index = LF_TOPO_NIC_INDEX(nicp->n_topo_index, p);
	topo_link->node_index_32[0] = htonl(node_index);

	/* fill in node_index[1] with link partner */
	onicp = LF_NIC(onp);
	node_index = LF_TOPO_NIC_INDEX(onicp->n_topo_index,
					nicp->topo_rports[p]);
	topo_link->node_index_32[1] = htonl(node_index);

	/* Save link state and topo index */
	topo_link_state[tl] = nicp->link_state[p];

	++topo_link;
	++tl;
      }
    }
  }

  if (tl != num_links) LF_ERROR(("link count mismatch!"));

  /*
   * Copy over link verifier information
   */
  topo_link = LF_TOPO_LINK_ARRAY(tm_buf);
  for (l=0; l<num_links; ++l) {
    int ni;
    int which;
    int port;
    int vni;
    struct lf_verifier *vp;

    /* Get to one end of link */
    ni = ntohl(topo_link[l].node_index_32[0]);

    /* If an xbar, check for a verifier */
    if (LF_TOPO_NI_XBAR(ni)) {
      x = LF_TOPO_NI_INDEX(ni);
      port = LF_TOPO_NI_PORT(ni);
      xp = fp->xbars[x];
      which = 0;

      /*
       * If verifier assigned, find the NIC it refers to and
       * save its map index in the link struct
       */
      vp = xp->verifiers + port;

      /* If the other end of the link is the designated verifier,
       * switch pointers to that xbar
       */
      if (vp->ver_nicp == LF_VERIFY_OTHER_END) {
	struct lf_xbar *xp2;
	xp2 = LF_XBAR(xp->topo_ports[port]);
	vp = xp2->verifiers + xp->topo_rports[port];
	which = 1;
      }

      nicp = vp->ver_nicp;
      if (nicp != NULL) {
	vni = LF_TOPO_VERIFIER_INDEX(nicp->n_topo_index,
	    				which, vp->ver_port);
	topo_link[l].verifier_32 = htonl(vni);
      }
    }
  }

  /*
   * Fill in LAG_IDs
   */
  for (l=0; l<fp->num_lag_ids; ++l) {
    lidp = &(fp->lag_ids[l]);
    lap = LF_TOPO_LAG_ID(topo_map, l);
    lf_lag_copy(lap, lidp->ld_lag_id);
  }

  *ret_map_size = topo_map_size;
  return topo_map;

 except:
  return NULL;
}
